<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:fn="fn"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:pl="http://product-live.com" version="3.0" exclude-result-prefixes="xs fn">
    
    <xsl:output method="xml" indent="yes" encoding="UTF-8"  />
    
    
    <!--
         + VERSION           : 1.0.3
         + TITLE             : Export zip medias
         + INPUT             : Items to export 
         + OUTPUT            : Generate archive xml file
         + PARAMETERS        :  
         + tableFile : Table taxonomy file
         + misc_languages : (string list, separated by commas) : Languages to use, PL codes
         + fields_to_export (string : ["all", "images", "attachments"]) or (string, list comma separated, fields key separated by commas)) : Setup the medias fields to export in the zips
         + folders (string : ["no_folders","one_folder_per_product", "one_folder_per_product_and_subfolder_per_media_type"])
         `      ->  no_folders : Directly output the files in the zip with no folders at all
         `      ->  one_folder_per_product : Create one folder for each product in the zip
         `      ->  one_folder_per_product_and_subfolder_per_media_type : Create one folder for each product and inside one subfolder for each media type
         
         + optional_images_resize_option (string) : Image resizing option, width and height in pixels separated by and underscore, exemple : 260_250 for 260 width and 250 height
         + optional_fieldKey_used_to_name_file: Optionnal Field or Identifier key used to name file and folders in the zip. By default the first non archived identifier will be used
         + optional_keep_original_fileName (boolean - YES/NO) : Keep the original media fileName in the export instead of renaming the file    
    -->
    
    <!--#region Parameters -->
    <xsl:param name="tableFile" select="'./TESTS/LH/table.xml'"/>
    
    <xsl:param name="fields_to_export" select="'all'" />
    <xsl:param name="folders" select="'one_folder_per_product'" />
    <xsl:param name="optional_images_resize_option" select="''" />
    <xsl:param name="optional_fieldKey_used_to_name_file" select="''" />
    <xsl:param name="optional_keep_original_fileName" select="''" />
    <xsl:param name="languages" select="''" />  
    <!--#endregion -->
    
    <xsl:variable name="_langs">
        <xsl:choose>
            <xsl:when test="normalize-space($languages) != '' and normalize-space($languages) != 'null'">
                <xsl:for-each select="tokenize($languages, ',')">
                    <lang>
                        <xsl:value-of select="."/>
                    </lang>
                </xsl:for-each>
            </xsl:when>
            <xsl:otherwise>
                <lang>default</lang>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:variable>
    <xsl:variable name="langs" select="$_langs/lang" />    
    <xsl:variable name="table" select="document($tableFile)/Table"></xsl:variable>
    <xsl:variable name="tableSchema" select="$table/Schema" />
    
    <!--Folders management-->
    <xsl:variable name="_folders" select="lower-case(normalize-space($folders))" />
    <xsl:variable name="folders_no_folders" select="$_folders = 'no_folders'" />
    <xsl:variable name="folders_one_folder_per_product" select="$_folders = 'one_folder_per_product'" />
    <xsl:variable name="folders_one_folder_per_product_and_subfolder_per_media_type" select="$_folders = 'one_folder_per_product_and_subfolder_per_media_type'" />
    
    <!--Creating optionnal images URL for resizing-->
    <xsl:variable name="image_url_suffix">
        <xsl:variable name="images_parameters">
            <map>
                <xsl:if test="$optional_images_resize_option != '' and $optional_images_resize_option != 'null'">
                    <xsl:variable name="width" select="tokenize($optional_images_resize_option, '_')[1]" />
                    <xsl:variable name="height" select="tokenize($optional_images_resize_option, '_')[2]" />
                    <xsl:if test="$width != '' and $width!='null' and number($width) > 0 ">
                        <entry key="width">
                            <xsl:value-of select="number($width)" />
                        </entry>
                    </xsl:if>
                    <xsl:if test="$height != '' and $height!='null' and number($height) > 0 ">
                        <entry key="height">
                            <xsl:value-of select="number($height)" />
                        </entry>
                    </xsl:if>
                </xsl:if>                
            </map>
        </xsl:variable>
        <xsl:variable name="attributes" select="$images_parameters/map/entry[text()!='']" />
        <xsl:if test="count($attributes)&gt;0">
            <xsl:text>?</xsl:text>
            <xsl:for-each select="$attributes">
                <xsl:choose>
                    <xsl:when test="position()&lt;count($attributes)">
                        <xsl:value-of select="concat(@key, '=', ., '&amp;')" />
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="concat(@key, '=', .)" />
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:for-each>
        </xsl:if>
    </xsl:variable>    
    
    <xsl:template match="/">
        <xsl:variable name="items" select="Table/Items/Item" />
        
        <xsl:variable name="_exportTableFields">
            <xsl:variable name="tokenizedExportOptions" select="tokenize($fields_to_export, ',')" />
            
            <xsl:variable name="all" select="$tokenizedExportOptions[normalize-space(upper-case(.)) = 'ALL'] or normalize-space($fields_to_export) = ''" />
            <xsl:variable name="images" select="$tokenizedExportOptions[normalize-space(upper-case(.)) = 'IMAGES'] or $all" />
            <xsl:variable name="attachments" select="$tokenizedExportOptions[normalize-space(upper-case(.)) = 'ATTACHMENTS'] or $all" />
            <xsl:variable name="specificFields" select="$tokenizedExportOptions[normalize-space(upper-case(.)) != 'ATTACHMENTS' and normalize-space(upper-case(.)) != 'ALL' and normalize-space(upper-case(.)) != 'IMAGES']" />
            <Fields>
                <xsl:variable name="notDistinctFields">
                    <xsl:if test="$images">
                        <xsl:copy-of select="$tableSchema/Fields/Field[@type = 'IMAGE']"/>  
                    </xsl:if>
                    <xsl:if test="$attachments">
                        <xsl:copy-of select="$tableSchema/Fields/Field[@type = 'ATTACHMENT']"/>   
                    </xsl:if>
                    <xsl:if test="count($specificFields) &gt; 0">
                        <xsl:copy-of select="$tableSchema/Fields/Field[$specificFields = @key ]"/> 
                    </xsl:if>
                </xsl:variable>
                <xsl:for-each-group select="$notDistinctFields/Field" group-by="@key">
                    <xsl:copy-of select="current-group()[1]"/>
                </xsl:for-each-group>
            </Fields>
        </xsl:variable>
        <xsl:variable name="exportTableFields" select="$_exportTableFields/Fields/Field" />        
        
        <Zips>
            <xsl:for-each select="$langs">
                <xsl:variable name="lang" select="." />
                <Zip>
                    <File-Name>
                        <xsl:variable name="date" select="format-dateTime(current-dateTime(), '[M01]-[D01]-[Y0001]_[H01]h[m01]m[s01]s')" />
                        <xsl:variable name="langText" select="if(normalize-space($lang) != 'default') then concat('_',upper-case($lang)) else ''" />
                        <xsl:value-of select="string-join( ($date,'_export_medias', $langText,   '.zip') , '')" />
                    </File-Name>
                    <Files>
                        <xsl:for-each select="$items">
                            <xsl:variable name="item" select="."/>
                            <xsl:variable name="itemFieldsImages" select="$item/Field[$exportTableFields[@type='IMAGE']/@key = @key]" />
                            <xsl:variable name="itemFieldsAttachments" select="$item/Field[$exportTableFields[@type='ATTACHMENT']/@key = @key]" />
                            
                            <xsl:if test="count($itemFieldsImages) > 0  or count($itemFieldsAttachments) > 0 ">
                                <!--Name of the current item folder-->
                                <xsl:variable name="baseItemPath">
                                    <xsl:choose>
                                        <xsl:when test="$folders_one_folder_per_product or $folders_one_folder_per_product_and_subfolder_per_media_type">
                                            <xsl:variable name="folderPath" select="pl:getItemFileName($item, $table)" />
                                            <xsl:choose>
                                                <xsl:when test="normalize-space($folderPath) != ''">
                                                    <xsl:value-of select="concat('/', $folderPath, '/')" />
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="'/NO_IDENTIFIER/'" />
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:text>/</xsl:text>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                
                                <!--Write Images fields-->
                                <xsl:if test="count($itemFieldsImages) > 0 ">                                 
                                    <xsl:variable name="imageBaseFilePath" >
                                        <xsl:value-of select="$baseItemPath"/>
                                        <xsl:if test="$folders_one_folder_per_product_and_subfolder_per_media_type">
                                            <xsl:value-of select="concat(if(pl:getLastChar($baseItemPath) = '/') then '' else '/', 'Images/')"/>
                                        </xsl:if>
                                    </xsl:variable>
                                    <xsl:call-template name="writeFields">
                                        <xsl:with-param name="exportTableFields" select = "$exportTableFields" />                                    
                                        <xsl:with-param name="item" select = "$item" />
                                        <xsl:with-param name="itemFields" select = "$itemFieldsImages" />
                                        <xsl:with-param name="baseFilePath" select = "$imageBaseFilePath" />
                                        <xsl:with-param name="urlSuffix" select = "$image_url_suffix" />
                                        <xsl:with-param name="lang" select = "$lang" />
                                    </xsl:call-template>
                                </xsl:if>
                                
                                <!--Write Fichiers fields-->
                                <xsl:if test="count($itemFieldsAttachments) > 0 ">                           
                                    <xsl:variable name="attachmentsBaseFilePath" >
                                        <xsl:value-of select="$baseItemPath"/>
                                        <xsl:if test="$folders_one_folder_per_product_and_subfolder_per_media_type">
                                            <xsl:value-of select="concat(if(pl:getLastChar($baseItemPath) = '/') then '' else '/', 'Documents/')"/>
                                        </xsl:if>
                                    </xsl:variable>
                                    <xsl:call-template name="writeFields">
                                        <xsl:with-param name="exportTableFields" select = "$exportTableFields" />  
                                        <xsl:with-param name="item" select = "$item" />
                                        <xsl:with-param name="itemFields" select = "$itemFieldsAttachments" />
                                        <xsl:with-param name="baseFilePath" select = "$attachmentsBaseFilePath" />
                                        <xsl:with-param name="lang" select = "$lang" />
                                        <xsl:with-param name="urlSuffix" select = "''" />
                                    </xsl:call-template>
                                </xsl:if>
                                
                            </xsl:if>
                        </xsl:for-each>
                    </Files>
                </Zip>
            </xsl:for-each>            
        </Zips>
    </xsl:template>
    
    <xsl:template name="writeFields">
        <xsl:param name = "item" />
        <xsl:param name = "itemFields" />
        <xsl:param name = "exportTableFields" />
        <xsl:param name = "baseFilePath" />
        <xsl:param name = "urlSuffix" />
        <xsl:param name = "lang" />
        
        <xsl:merge>
            <xsl:merge-source name="itemField" select="$itemFields" sort-before-merge="yes">
                <xsl:merge-key select="@key"/>
            </xsl:merge-source>
            <xsl:merge-source name="tableField" select="$exportTableFields" sort-before-merge="yes">
                <xsl:merge-key select="@key"/>
            </xsl:merge-source>
            <xsl:merge-action>
                <xsl:variable name="itemField" select="current-merge-group('itemField')" />
                <xsl:variable name="tableField" select="current-merge-group('tableField')" />
                <xsl:if test="$itemField and $tableField">
                    <xsl:variable name="fieldType" select="$tableField/@type" />
                    <File>                        
                        <!--Building full file path -->
                        <xsl:variable name="fullFilePath">
                            <!--BasePath-->
                            <xsl:value-of select="$baseFilePath"></xsl:value-of>                            
                            <xsl:choose>
                                <!--Keeping the original file-name-->
                                <xsl:when test="upper-case(normalize-space($optional_keep_original_fileName)) = 'YES'">
                                    <xsl:value-of select="$itemField/@original-file-name"/>
                                    <!--In case the original filename don't have the extension-->
                                    <xsl:if test="not(contains($itemField/@original-file-name,'.'))">
                                        <xsl:value-of select="concat('.',pl:getFileExtension($itemField))"/>
                                    </xsl:if>
                                </xsl:when>
                                <!--Renaming file-->
                                <xsl:otherwise>
                                    <!--Base file name -->
                                    <xsl:value-of select="pl:getItemFileName($item, $table)" />
                                    
                                    <!--Adding Field title-->
                                    <xsl:value-of select="concat('-', pl:getBanalisedFileName(pl:getLocalizedTitle($tableField,$lang)))" />
                                    
                                    <!--Adding file extension -->
                                    <xsl:value-of select="concat('.', pl:getFileExtension($itemField))" />
                                </xsl:otherwise>
                            </xsl:choose>                            
                        </xsl:variable>
                        
                        <Path>
                            <xsl:value-of select="$fullFilePath" />
                        </Path>
                        <Url>
                            <xsl:choose>
                                <xsl:when test="$fieldType = 'IMAGE' and $urlSuffix != ''">
                                    <xsl:value-of select="concat(., $urlSuffix)" />
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="." />
                                </xsl:otherwise>
                            </xsl:choose>
                        </Url>
                    </File>
                </xsl:if>
            </xsl:merge-action>
        </xsl:merge>
    </xsl:template>
    
    <xsl:function name="pl:getLastChar">
        <xsl:param name="value"  />
        <xsl:if test="normalize-space($value) != ''">
            <xsl:value-of select="substring($value, string-length($value), 1)" />
        </xsl:if>        
    </xsl:function>
    
    <xsl:function name="pl:getLocalizedTitle">
        <xsl:param name="element" />
        <xsl:param name="lang" />
        
        <xsl:variable name="baseTitle" select="$element/Title" />
        <xsl:choose>
            <xsl:when test="not($lang) or normalize-space($lang) = '' or normalize-space($lang) = 'default'">
                <xsl:value-of select="$baseTitle"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="foundLang" select="$element/Title-Local[upper-case(normalize-space(@lang)) = upper-case(normalize-space($lang))]" />
                <xsl:value-of select="if($foundLang and normalize-space($foundLang) != '') then $foundLang else $baseTitle"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <xsl:function name="pl:getItemFileName">
        <xsl:param name="item" />
        <xsl:param name="table" />
        <xsl:value-of select="pl:getBanalisedFileName(pl:getItemFieldKeyUsedToNameFile($item,$table))" />
    </xsl:function>
    
    
    <xsl:function name="pl:getItemFieldKeyUsedToNameFile">
        <xsl:param name="item" />
        <xsl:param name="table" />
        
        <xsl:variable name="availableValues">
            <xsl:if test="normalize-space($optional_fieldKey_used_to_name_file) != ''">
                <xsl:variable name="itemValue" select="$item/Identifier[@key = $optional_fieldKey_used_to_name_file] | $item/Field[@key = $optional_fieldKey_used_to_name_file]" />
                <xsl:if test="$itemValue">
                    <value priority="1">
                        <xsl:value-of select="$itemValue"/>
                    </value>
                </xsl:if>
            </xsl:if>
            <xsl:for-each select="$tableSchema/Identifiers/Identifier[not(@status = 'ARCHIVED')]">
                <xsl:sort select="@index" order="ascending" data-type="number" />
                <xsl:variable name="key" select="@key" />
                <xsl:variable name="itemValue" select="$item/Identifier[@key= $key]" />
                <xsl:if test="$itemValue">
                    <value priority="{position()+1}">
                        <xsl:value-of select="$itemValue"/>
                    </value>
                </xsl:if>
            </xsl:for-each>
            <value priority="50">NO_IDENTIFIER</value>
        </xsl:variable>
        <xsl:variable name="value">
            <xsl:for-each select="$availableValues/value">
                <xsl:sort select="@priority" order="ascending" data-type="number" />
                <xsl:if test="position() = 1">
                    <xsl:value-of select="."/>
                </xsl:if>
            </xsl:for-each>
        </xsl:variable>
        <xsl:value-of select="pl:getBanalisedFileName($value)" />
    </xsl:function>
    
    <!--Version 1.0-->
    <xsl:function name="pl:getBanalisedFileName">
        <xsl:param name="value" />
        <xsl:value-of select="normalize-space(replace(replace(replace(normalize-unicode($value,'NFKD'),' ','_'), '/', '_'), '[^A-Za-z0-9.@:\-_ ]', ''))" />
    </xsl:function>
    
    <xsl:function name="pl:getFileExtension">
        <xsl:param name="field" />
        
        <xsl:variable name="mimeType" select="$field/@mime-type" />        
        <xsl:if test="$mimeType and normalize-space($mimeType) != ''">
            <xsl:variable name="mimeTypeToLower" select="lower-case($mimeType)" />
            <xsl:choose>
                <xsl:when test="$mimeTypeToLower = 'text/csv'">
                    <xsl:value-of select="'csv'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/pdf'">
                    <xsl:value-of select="'pdf'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'">
                    <xsl:value-of select="'docx'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'">
                    <xsl:value-of select="'xlsx'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/vnd.ms-powerpoint'">
                    <xsl:value-of select="'ppt'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/vnd.openxmlformats-officedocument.presentationml.presentation'">
                    <xsl:value-of select="'pptx'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'audio/x-wav'">
                    <xsl:value-of select="'wav'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'video/mp4'">
                    <xsl:value-of select="'mp4'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'video/mpeg'">
                    <xsl:value-of select="'mpeg'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'audio/mpeg'">
                    <xsl:value-of select="'mp3'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'video/x-msvideo'">
                    <xsl:value-of select="'avi'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'image/png'">
                    <xsl:value-of select="'png'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'image/jpeg'">
                    <xsl:value-of select="'jpg'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'image/x-dwg' or $mimeTypeToLower='image/vnd.dwg'">
                    <xsl:value-of select="'dwg'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'image/svg+xml'">
                    <xsl:value-of select="'svg'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'image/tiff'">
                    <xsl:value-of select="'tiff'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/gzip'">
                    <xsl:value-of select="'gz'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = ' application/zip'">
                    <xsl:value-of select="'zip'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'text/html'">
                    <xsl:value-of select="'html'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/json'">
                    <xsl:value-of select="'json'" />
                </xsl:when>
                <xsl:when test="$mimeTypeToLower = 'application/cdfv2'">
                    <xsl:value-of select="'rfa'" />
                </xsl:when>
                <xsl:otherwise>
                    <!--
                         mime-type unknown or equal to application/octet-stream 
                         => We try to detect the file extension with all the infos we can get 
                    -->
                    <xsl:choose>
                        <xsl:when test="not(contains($field/@original-file-name, 'https://')) and contains($field/@original-file-name, '.')">
                            <xsl:value-of select="tokenize($field/@original-file-name, '\.')[last()]"/>
                        </xsl:when>
                        <xsl:when test="not(contains($field/@file-key, 'https://')) and contains($field/@file-key, '.')">
                            <xsl:value-of select="tokenize($field/@file-key, '\.')[last()]"/>
                        </xsl:when>
                        <xsl:when test="not(contains($field/@file-hash, 'https://')) and contains($field/@file-hash, '.')">
                            <xsl:value-of select="tokenize($field/@file-hash, '\.')[last()]"/>
                        </xsl:when>
                    </xsl:choose>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:if>
    </xsl:function>
    
</xsl:stylesheet>